<?php

namespace WeStacks\TeleBot\Laravel\Notifications;

use JsonSerializable;
use WeStacks\TeleBot\Exceptions\TeleBotException;
use WeStacks\TeleBot\Traits\HasTelegramMethods;

/**
 * This class represents a bot instance. This is basically main controller for sending and handling your Telegram requests.
 *
 * @method static self addStickerToSet(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to add a new sticker to a set created by the bot. You must use exactly one of the fields png_sticker or tgs_sticker. Animated stickers can be added to animated sticker sets and only to them. Animated sticker sets can have up to 50 stickers. Static sticker sets can have up to 120 stickers. Returns True on success.
 * @method static self answerCallbackQuery(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           Use this method to send answers to callback queries sent from [inline keyboards](/bots#inline-keyboards-and-on-the-fly-updating). The answer will be displayed to the user as a notification at the top of the chat screen or as an alert. On success, True is returned.
 * @method static self answerInlineQuery(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             Use this method to send answers to an inline query. On success, True is returned.
 * @method static self answerPreCheckoutQuery(array $parameters = []) Once the user has confirmed their payment and shipping details, the Bot API sends the final confirmation in the form of an [Update](https://core.telegram.org/bots/api#update)                                                                                                                                                                                                                                                                                                                                                                         with the field pre_checkout_query. Use this method to respond to such pre-checkout queries. On success, True is returned. Note: The Bot API must receive an answer within 10 seconds after the pre-checkout query was sent.
 * @method static self answerShippingQuery(array $parameters = []) If you sent an invoice requesting a shipping address and the parameter is_flexible was specified, the Bot API will send an [Update](https://core.telegram.org/bots/api#update)                                                                                                                                                                                                                                                                                                                                                                            with a shipping_query field to the bot. Use this method to reply to shipping queries. On success, True is returned.
 * @method static self answerWebAppQuery(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             Use this method to set the result of an interaction with a Web App and send a corresponding message on behalf of the user to the chat from which the query originated. On success, a SentWebAppMessage object is returned.
 * @method static self approveChatJoinRequest(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        Use this method to approve a chat join request. The bot must be an administrator in the chat for this to work and must have the can_invite_users administrator right. Returns True on success.
 * @method static self banChatMember(array $parameters = []) Use this method to ban a user in a group, a supergroup or a channel. In the case of supergroups and channels, the user will not be able to return to the chat on their own using invite links, etc., unless [unbanned](https://core.telegram.org/bots/api#unbanchatmember)                                                                                                                                                                                                                                                                                      first. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self banChatSenderChat(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             Use this method to ban a channel chat in a supergroup or a channel. Until the chat is [unbanned](https://core.telegram.org/bots/api#unbanchatsenderchat), the owner of the banned chat won't be able to send messages on behalf of any of their channels. The bot must be an administrator in the supergroup or channel for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self close()                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to close the bot instance before moving it from one local server to another. You need to delete the webhook before calling this method to ensure that the bot isn't launched again after server restart. The method will return error 429 in the first 10 minutes after the bot is launched. Returns True on success. Requires no parameters.
 * @method static self copyMessage(array $parameters = []) Use this method to copy messages of any kind. Service messages and invoice messages can't be copied. The method is analogous to the method [forwardMessage](https://core.telegram.org/bots/api#forwardmessage), but the copied message doesn't have a link to the original message. Returns the [MessageId](https://core.telegram.org/bots/api#messageid)                                                                                                                                                                                                         of the sent message on success.
 * @method static self createChatInviteLink(array $parameters = []) Use this method to create an additional invite link for a chat. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. The link can be revoked using the method [revokeChatInviteLink](https://core.telegram.org/bots/api#revokechatinvitelink). Returns the new invite link as [ChatInviteLink](https://core.telegram.org/bots/api#chatinvitelink)                                                                                                                                           object.
 * @method static self createInvoiceLink(array $parameters = []) Use this method to create a link for an invoice. Returns the created invoice link as String on success.
 * @method static self createNewStickerSet(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           Use this method to create a new sticker set owned by a user. The bot will be able to edit the sticker set thus created. You must use exactly one of the fields png_sticker or tgs_sticker. Returns True on success.
 * @method static self declineChatJoinRequest(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        Use this method to decline a chat join request. The bot must be an administrator in the chat for this to work and must have the can_invite_users administrator right. Returns True on success.
 * @method static self deleteChatPhoto(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to delete a chat photo. Photos can't be changed for private chats. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self deleteChatStickerSet(array $parameters = []) Use this method to delete a group sticker set from a supergroup. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Use the field can_set_sticker_set optionally returned in [getChat](https://core.telegram.org/bots/api#getchat)                                                                                                                                                                                                                                                        requests to check if the bot can use this method. Returns True on success.
 * @method static self getForumTopicIconStickers() Use this method to get custom emoji stickers, which can be used as a forum topic icon by any user. Requires no parameters. Returns an Array of Sticker objects.
 * @method static self createForumTopic(array $parameters = []) Use this method to create a topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns information about the created topic as a ForumTopic object.
 * @method static self editForumTopic(array $parameters = []) Use this method to edit name and icon of a topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have can_manage_topics administrator rights, unless it is the creator of the topic. Returns True on success.
 * @method static self closeForumTopic(array $parameters = []) Use this method to close an open topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights, unless it is the creator of the topic. Returns True on success.
 * @method static self reopenForumTopic(array $parameters = []) Use this method to reopen a closed topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights, unless it is the creator of the topic. Returns True on success.
 * @method static self deleteForumTopic(array $parameters = []) Use this method to delete a forum topic along with all its messages in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_delete_messages administrator rights. Returns True on success.
 * @method static self unpinAllForumTopicMessages(array $parameters = []) Use this method to clear the list of pinned messages in a forum topic. The bot must be an administrator in the chat for this to work and must have the can_pin_messages administrator right in the supergroup. Returns True on success.
 * @method static self editGeneralForumTopic(array $parameters = []) Use this method to edit the name of the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have can_manage_topics administrator rights. Returns True on success.
 * @method static self editGeneralForumTopic(array $parameters = []) Use this method to edit the name of the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have can_manage_topics administrator rights. Returns True on success.
 * @method static self closeGeneralForumTopic(array $parameters = []) Use this method to close an open 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns True on success.
 * @method static self reopenGeneralForumTopic(array $parameters = []) Use this method to reopen a closed 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. The topic will be automatically unhidden if it was hidden. Returns True on success.
 * @method static self hideGeneralForumTopic(array $parameters = []) Use this method to hide the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. The topic will be automatically closed if it was open. Returns True on success.
 * @method static self unhideGeneralForumTopic(array $parameters = []) Use this method to unhide the 'General' topic in a forum supergroup chat. The bot must be an administrator in the chat for this to work and must have the can_manage_topics administrator rights. Returns True on success.
 * @method static self unpinAllGeneralForumTopicMessages(array $parameters = []) Use this method to clear the list of pinned messages in a General forum topic. The bot must be an administrator in the chat for this to work and must have the can_pin_messages administrator right in the supergroup. Returns True on success.
 * @method static self deleteMessage(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 Use this method to delete a message, including service messages, with the following limitations:
 * @method static self deleteMyCommands(array $parameters = []) Use this method to delete the list of the bot's commands for the given scope and user language. After deletion, [higher level commands](https://core.telegram.org/bots/api#determining-list-of-commands)                                                                                                                                                                                                                                                                                                                                                     will be shown to affected users. Returns True on success.
 * @method static self deleteStickerFromSet(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          Use this method to delete a sticker from a set created by the bot. Returns True on success.
 * @method static self deleteWebhook(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 Use this method to remove webhook integration if you decide to switch back to [getUpdates](https://core.telegram.org/bots/api#getupdates). Returns True on success.
 * @method static self editChatInviteLink(array $parameters = []) Use this method to edit a non-primary invite link created by the bot. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns the edited invite link as a [ChatInviteLink](https://core.telegram.org/bots/api#chatinvitelink)                                                                                                                                                                                                                                                            object.
 * @method static self editMessageCaption(array $parameters = []) Use this method to edit captions of messages. On success, if the edited message is not an inline message, the edited [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                 is returned, otherwise True is returned.
 * @method static self editMessageLiveLocation(array $parameters = []) Use this method to edit live location messages. A location can be edited until its live_period expires or editing is explicitly disabled by a call to [stopMessageLiveLocation](https://core.telegram.org/bots/api#stopmessagelivelocation). On success, if the edited message is not an inline message, the edited [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                             is returned, otherwise True is returned.
 * @method static self editMessageMedia(array $parameters = []) Use this method to edit animation, audio, document, photo, or video messages. If a message is part of a message album, then it can be edited only to an audio for audio albums, only to a document for document albums and to a photo or a video otherwise. When an inline message is edited, a new file can't be uploaded; use a previously uploaded file via its file_id or specify a URL. On success, if the edited message is not an inline message, the edited [Message](https://core.telegram.org/bots/api#message)                                    is returned, otherwise True is returned.
 * @method static self editMessageReplyMarkup(array $parameters = []) Use this method to edit only the reply markup of messages. On success, if the edited message is not an inline message, the edited [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                is returned, otherwise True is returned.
 * @method static self editMessageText(array $parameters = []) Use this method to edit text and [game](https://core.telegram.org/bots/api#games) messages. On success, if the edited message is not an inline message, the edited [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                      is returned, otherwise True is returned.
 * @method static self exportChatInviteLink(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          Use this method to generate a new primary invite link for a chat; any previously generated primary link is revoked. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns the new invite link as String on success.
 * @method static self forwardMessage(array $parameters = []) Use this method to forward messages of any kind. Service messages can't be forwarded. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                               is returned.
 * @method static self getChatAdministrators(array $parameters = []) Use this method to get a list of administrators in a chat. On success, returns an Array of [ChatMember](https://core.telegram.org/bots/api#chatmember)                                                                                                                                                                                                                                                                                                                                                                                                  objects that contains information about all chat administrators except other bots. If the chat is a group or a supergroup and no administrators were appointed, only the creator will be returned.
 * @method static self getChatMemberCount(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            Use this method to get the number of members in a chat. Returns Int on success.
 * @method static self getChatMember(array $parameters = []) Use this method to get information about a member of a chat. Returns a [ChatMember](https://core.telegram.org/bots/api#chatmember)                                                                                                                                                                                                                                                                                                                                                                                                                              object on success.
 * @method static self getChatMenuButton(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             Use this method to get the current value of the bot's menu button in a private chat, or the default menu button. Returns MenuButton on success.
 * @method static self getChat(array $parameters = []) Use this method to get up to date information about the chat (current name of the user for one-on-one conversations, current username of a user, group or channel, etc.). Returns a [Chat](https://core.telegram.org/bots/api#chat)                                                                                                                                                                                                                                                                                                                                   object on success.
 * @method static self getCustomEmojiStickers(array $parameters = []) Use this method to get information about custom emoji stickers by their identifiers. Returns an Array of Sticker objects.
 * @method static self getFile(array $parameters = []) Use this method to get basic info about a file and prepare it for downloading. For the moment, bots can download files of up to 20MB in size. On success, a [File](https://core.telegram.org/bots/api#file) object is returned. The file can then be downloaded via the link https://api.telegram.org/file/bot<token>/<file_path>, where <file_path> is taken from the response. It is guaranteed that the link will be valid for at least 1 hour. When the link expires, a new one can be requested by calling [getFile](https://core.telegram.org/bots/api#getfile) again.
 * @method static self getGameHighScores(array $parameters = []) Use this method to get data for high score tables. Will return the score of the specified user and several of their neighbors in a game. On success, returns an Array of [GameHighScore](https://core.telegram.org/bots/api#gamehighscore)                                                                                                                                                                                                                                                                                                                  objects.
 * @method static self getMe() A simple method for testing your bot's authentication token. Requires no parameters. Returns basic information about the bot in form of a [User](https://core.telegram.org/bots/api#user)                                                                                                                                                                                                                                                                                                                                                                                                     object.
 * @method static self getMyCommands(array $parameters = []) Use this method to get the current list of the bot's commands for the given scope and user language. Returns Array of [BotCommand](https://core.telegram.org/bots/api#botcommand)                                                                                                                                                                                                                                                                                                                                                                               on success. If commands aren't set, an empty list is returned.
 * @method static self getMyDefaultAdministratorRights(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to get the current default administrator rights of the bot. Returns ChatAdministratorRights on success.
 * @method static self getStickerSet(array $parameters = []) Use this method to get a sticker set. On success, a [StickerSet](https://core.telegram.org/bots/api#stickerset)                                                                                                                                                                                                                                                                                                                                                                                                                                                 object is returned.
 * @method static self getUpdates(array $parameters = []) Use this method to receive incoming updates using long polling ([wiki](https://en.wikipedia.org/wiki/Push_technology#Long_polling)). An Array of [Update](https://core.telegram.org/bots/api#update)                                                                                                                                                                                                                                                                                                                                                               objects is returned.
 * @method static self getUserProfilePhotos(array $parameters = []) Use this method to get a list of profile pictures for a user. Returns a [UserProfilePhotos](https://core.telegram.org/bots/api#userprofilephotos)                                                                                                                                                                                                                                                                                                                                                                                                        object.
 * @method static self getWebhookInfo() Use this method to get current webhook status. Requires no parameters. On success, returns a [WebhookInfo](https://core.telegram.org/bots/api#webhookinfo)                                                                                                                                                                                                                                                                                                                                                                                                                           object. If the bot is using [getUpdates](https://core.telegram.org/bots/api#getupdates), will return an object with the url field empty.
 * @method static self leaveChat(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     Use this method for your bot to leave a group, supergroup or channel. Returns True on success.
 * @method static self logOut()                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              Use this method to log out from the cloud Bot API server before launching the bot locally. You must log out the bot before running it locally, otherwise there is no guarantee that the bot will receive updates. After a successful call, you can immediately log in on a local server, but will not be able to log in back to the cloud Bot API server for 10 minutes. Returns True on success. Requires no parameters.
 * @method static self pinChatMessage(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                Use this method to add a message to the list of pinned messages in a chat. If the chat is not a private chat, the bot must be an administrator in the chat for this to work and must have the 'can_pin_messages' administrator right in a supergroup or 'can_edit_messages' administrator right in a channel. Returns True on success.
 * @method static self promoteChatMember(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             Use this method to promote or demote a user in a supergroup or a channel. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Pass False for all boolean parameters to demote a user. Returns True on success.
 * @method static self restrictChatMember(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            Use this method to restrict a user in a supergroup. The bot must be an administrator in the supergroup for this to work and must have the appropriate administrator rights. Pass True for all permissions to lift restrictions from a user. Returns True on success.
 * @method static self revokeChatInviteLink(array $parameters = []) Use this method to revoke an invite link created by the bot. If the primary link is revoked, a new link is automatically generated. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns the revoked invite link as [ChatInviteLink](https://core.telegram.org/bots/api#chatinvitelink)                                                                                                                                                                                             object.
 * @method static self sendAnimation(array $parameters = []) Use this method to send animation files (GIF or H.264/MPEG-4 AVC video without sound). On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                               is returned. Bots can currently send animation files of up to 50 MB in size, this limit may be changed in the future.
 * @method static self sendAudio(array $parameters = []) Use this method to send audio files, if you want Telegram clients to display them in the music player. Your audio must be in the .MP3 or .M4A format. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                    is returned. Bots can currently send audio files of up to 50 MB in size, this limit may be changed in the future.
 * @method static self sendChatAction(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                Use this method when you need to tell the user that something is happening on the bot's side. The status is set for 5 seconds or less (when a message arrives from your bot, Telegram clients clear its typing status). Returns True on success.
 * @method static self sendContact(array $parameters = []) Use this method to send phone contacts. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                                is returned.
 * @method static self sendDice(array $parameters = []) Use this method to send an animated emoji that will display a random value. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                               is returned.
 * @method static self sendDocument(array $parameters = []) Use this method to send general files. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                                is returned. Bots can currently send files of any type of up to 50 MB in size, this limit may be changed in the future.
 * @method static self sendGame(array $parameters = []) Use this method to send a game. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                                           is returned.
 * @method static self sendInvoice(array $parameters = []) Use this method to send invoices. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                                      is returned.
 * @method static self sendLocation(array $parameters = []) Use this method to send point on the map. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                             is returned.
 * @method static self sendMediaGroup(array $parameters = []) Use this method to send a group of photos, videos, documents or audios as an album. Documents and audio files can be only grouped in an album with messages of the same type. On success, an array of [Messages](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                   that were sent is returned.
 * @method static self sendMessage(array $parameters = []) Use this method to send text messages. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                                 is returned.
 * @method static self sendPoll(array $parameters = []) Use this method to send a native poll. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                                    is returned.
 * @method static self sendSticker(array $parameters = []) Use this method to send static .WEBP or [animated](https://telegram.org/blog/animated-stickers) .TGS stickers. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                         is returned.
 * @method static self sendVenue(array $parameters = []) Use this method to send information about a venue. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                                                                                       is returned.
 * @method static self sendVideo(array $parameters = []) Use this method to send video files, Telegram clients support mp4 videos (other formats may be sent as [Document](https://core.telegram.org/bots/api#document)). On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                         is returned. Bots can currently send video files of up to 50 MB in size, this limit may be changed in the future.
 * @method static self sendVideoNote(array $parameters = []) As of [v.4.0](https://telegram.org/blog/video-messages-and-telescope), Telegram clients support rounded square mp4 videos of up to 1 minute long. Use this method to send video messages. On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                            is returned.
 * @method static self sendVoice(array $parameters = []) Use this method to send audio files, if you want Telegram clients to display the file as a playable voice message. For this to work, your audio must be in an .OGG file encoded with OPUS (other formats may be sent as [Audio](https://core.telegram.org/bots/api#audio) or [Document](https://core.telegram.org/bots/api#document)). On success, the sent [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                   is returned. Bots can currently send voice messages of up to 50 MB in size, this limit may be changed in the future.
 * @method static self setChatAdministratorCustomTitle(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to set a custom title for an administrator in a supergroup promoted by the bot. Returns True on success.
 * @method static self setChatDescription(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            Use this method to change the description of a group, a supergroup or a channel. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self setChatMenuButton(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             Use this method to change the description of a group, a supergroup or a channel. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self setChatPermissions(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            Use this method to set default chat permissions for all members. The bot must be an administrator in the group or a supergroup for this to work and must have the can_restrict_members administrator rights. Returns True on success.
 * @method static self setChatPhoto(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  Use this method to set a new profile photo for the chat. Photos can't be changed for private chats. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self setChatStickerSet(array $parameters = []) Use this method to set a new group sticker set for a supergroup. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Use the field can_set_sticker_set optionally returned in [getChat](https://core.telegram.org/bots/api#getchat)                                                                                                                                                                                                                                                           requests to check if the bot can use this method. Returns True on success.
 * @method static self setChatTitle(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  Use this method to change the title of a chat. Titles can't be changed for private chats. The bot must be an administrator in the chat for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self setGameScore(array $parameters = []) Use this method to set the score of the specified user in a game message. On success, if the message is not an inline message, the [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                                                         is returned, otherwise True is returned. Returns an error, if the new score is not greater than the user's current score in the chat and force is False.
 * @method static self setMyCommands(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 Use this method to change the list of the bot's commands. See https://core.telegram.org/bots#commands for more details about bot commands. Returns True on success.
 * @method static self setMyDefaultAdministratorRights(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to change the default administrator rights requested by the bot when it's added as an administrator to groups or channels. These rights will be suggested to users, but they are are free to modify the list before adding the bot. Returns True on success.
 * @method static self setPassportDataErrors(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         Informs a user that some of the Telegram Passport elements they provided contains errors. The user will not be able to re-submit their Passport to you until the errors are fixed (the contents of the field for which you returned the error must change). Returns True on success.
 * @method static self setStickerPositionInSet(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       Use this method to move a sticker in a set created by the bot to a specific position. Returns True on success.
 * @method static self setStickerSetThumb(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            Use this method to set the thumbnail of a sticker set. Animated thumbnails can be set for animated sticker sets only. Returns True on success.
 * @method static self setWebhook(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    Use this method to specify a url and receive incoming updates via an outgoing webhook. Whenever there is an update for the bot, we will send an HTTPS POST request to the specified url, containing a JSON-serialized [Update](https://core.telegram.org/bots/api#update). In case of an unsuccessful request, we will give up after a reasonable amount of attempts. Returns True on success.
 * @method static self stopMessageLiveLocation(array $parameters = []) Use this method to stop updating a live location message before live_period expires. On success, if the message is not an inline message, the edited [Message](https://core.telegram.org/bots/api#message)                                                                                                                                                                                                                                                                                                                                            is returned, otherwise True is returned.
 * @method static self stopPoll(array $parameters = []) Use this method to stop a poll which was sent by the bot. On success, the stopped [Poll](https://core.telegram.org/bots/api#poll)                                                                                                                                                                                                                                                                                                                                                                                                                                    is returned.
 * @method static self unbanChatMember(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               Use this method to unban a previously banned user in a supergroup or channel. The user will not return to the group or channel automatically, but will be able to join via link, etc. The bot must be an administrator for this to work. By default, this method guarantees that after the call the user is not a member of the chat, but will be able to join it. So if the user is a member of the chat they will also be removed from the chat. If you don't want this, use the parameter only_if_banned. Returns True on success.
 * @method static self unbanChatSenderChat(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           Use this method to unban a previously banned channel chat in a supergroup or channel. The bot must be an administrator for this to work and must have the appropriate administrator rights. Returns True on success.
 * @method static self unpinAllChatMessages(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          Use this method to clear the list of pinned messages in a chat. If the chat is not a private chat, the bot must be an administrator in the chat for this to work and must have the 'can_pin_messages' administrator right in a supergroup or 'can_edit_messages' administrator right in a channel. Returns True on success.
 * @method static self unpinChatMessage(array $parameters = [])                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              Use this method to remove a message from the list of pinned messages in a chat. If the chat is not a private chat, the bot must be an administrator in the chat for this to work and must have the 'can_pin_messages' administrator right in a supergroup or 'can_edit_messages' administrator right in a channel. Returns True on success.
 * @method static self uploadStickerFile(array $parameters = []) Use this method to upload a .PNG file with a sticker for later use in createNewStickerSet and addStickerToSet methods (can be used multiple times). Returns the uploaded [File](https://core.telegram.org/bots/api#file)                                                                                                                                                                                                                                                                                                                                    on success.
 */
class TelegramNotification implements JsonSerializable, \Stringable
{
    use HasTelegramMethods;

    protected $data;

    /**
     * Create new notification instance.
     *
     * @param  string|null  $data JSON data object representation
     */
    public function __construct(string $data = null)
    {
        $this->data = $data ? json_decode($data, true) : [
            'bot' => null,
            'actions' => [],
        ];
    }

    public static function __callStatic(string $name, array $arguments)
    {
        return (new static)->$name(...$arguments);
    }

    public function __call(string $method, array $arguments)
    {
        if (! static::method($method)) {
            throw new TeleBotException('Method "'.$method.'" does not exist.');
        }

        $this->data['actions'][] = [
            'method' => $method,
            'arguments' => is_array($arguments[0]) ? $arguments[0] : [],
        ];

        return $this;
    }

    /**
     * Set bot to send notification.
     *
     * @return self
     */
    public function bot(string $bot)
    {
        $this->data['bot'] = $bot;

        return $this;
    }

    #[\ReturnTypeWillChange]
    public function jsonSerialize()
    {
        return $this->data;
    }

    public function __toString(): string
    {
        return (string) json_encode($this->data);
    }
}
